# Milight Multi-Bridge Emulator
# Version 0.1
# Cree le 29/07/2016
# Jerome Michaux (contact _at_ g-rom.info)
# Usage et diffusion libre sous reserve de citation de l'origine des sources
# www.g-rom.info

class MilightRemote:

    #Constructeur
    def __init__(self, remoteid, driver):
        try:
            assert remoteid > 0
            assert remoteid < 0xFFFF
            self._remoteid = remoteid
        except AssertionError:
            print("ERROR: RemoteID must be set between 0x0000 and 0xFFFE")
            self._remoteid = 0
        self._sequence = 0
        self.driver = driver

    def _get_sequence(self):
        return self._sequence

    def _get_sequence(self):
        return self._sequence

    def _set_sequence(self, new_sequence):
        self._sequence = new_sequence
        try:
            assert new_sequence >= 0
            assert new_sequence < 256
            self._sequence = new_sequence
        except AssertionError:
            print("ERROR: Sequence must be set between 0 and 255")

    sequence = property(_get_sequence, _set_sequence)

    #Fonction gérant l'incrémentation des numéros de séquence
    def increm_sequence(self):
        if self.sequence >= 255:
            self.sequence = 0
        else:
            self.sequence += 1

class MilightRemoteRGBW(MilightRemote):

    # Variables de la classe
    _COLOR_NAMES = {}
    _COLOR_NAMES[0] = "violet"
    _COLOR_NAMES[16] = "royalblue"
    _COLOR_NAMES[32] = "babyblue"
    _COLOR_NAMES[48] = "aqua"
    _COLOR_NAMES[64] = "mint"
    _COLOR_NAMES[80] = "seafoamgreen"
    _COLOR_NAMES[96] = "green"
    _COLOR_NAMES[112] = "limegreen"
    _COLOR_NAMES[128] = "yellow"
    _COLOR_NAMES[144] = "yelloworange"
    _COLOR_NAMES[160] = "orange"
    _COLOR_NAMES[176] = "red"
    _COLOR_NAMES[192] = "pink"
    _COLOR_NAMES[208] = "fuschia"
    _COLOR_NAMES[224] = "lilac"
    _COLOR_NAMES[240] = "lavender"

    #Constructeur vide
    def __init__(self, remoteid, driver):
        MilightRemote.__init__(self, remoteid, driver)
        self._color = [0, 0, 0, 0, 0]
        self._brightnessW = [26, 26, 26, 26, 26]
        self._brightnessRGB = [26, 26, 26, 26, 26]
        self._mode = 0
        self._command = 0
        self._group = 0
        self._discomode = 0xB8
        StrRemoteID = '{:06x}'.format(self._remoteid).upper()
        print("Creation of RGBW Remote : " + StrRemoteID)

    #Properties d'acces aux parametres color, brightness, command et group
    def _get_color(self):
        return self._color[self.group]

    def _set_color(self, new_color):
        try:
            assert new_color >= 0
            assert new_color < 256
            self._color[self.group] = new_color
            self._mode = 1
            self.command = 15
        except AssertionError:
            print("ERROR: Color must be set between 0 and 255")

    color = property(_get_color, _set_color)

    def _get_brightness(self):
        if self._mode == 0:
            return self._brightnessW[self.group]
        else:
            return self._brightnessRGB[self.group]

    def _set_brightness(self, new_brightness):
        #Valeur haute = 0xB8 (184)
        #Pas de 8
        #+1 pour le numero de groupe ?

        try:
            assert new_brightness >= 0
            assert new_brightness < 26
            if(self._mode == 0):
                self._brightnessW[self.group] = new_brightness
            else:
                self._brightnessRGB[self.group] = new_brightness
            self.command = 14

        except AssertionError:
            print("ERROR: Brightness must be set between 0 and 25")

    brightness = property(_get_brightness, _set_brightness)

    def _get_group(self):
        return self._group

    def _set_group(self, new_group):
        try:
            assert new_group >= 0
            assert new_group < 5
            self._group = new_group
        except AssertionError:
            print("ERROR: Group must be set between 0 and 4")

    group = property(_get_group, _set_group)

    def _get_command(self):
        return self._command

    def _set_command(self, new_command):
        '''
        Valeurs 02, 04, 06, 08, 10 = PowerOff All, Groupes 1, 2, 3, 4
        Vleurs 01, 03, 05, 07, 09 = PowerOn All, Groupes 1, 2, 3, 4
        Valeur 11 = Speed Up
        Valeur 12 = Speed Down
        Valeur 13 = Disco mode select
        Valeur 14 = Brightness
        Valeur 15 = Color
        Valeurs 18, 20, 22, 24, 26 = Passage en mode blanc, Groupes 1, 2, 3, 4
        Valeurs 17, 19, 21, 23, 25 = Passage en mode blanc, Groupes 1, 2, 3, 4
        '''
        try:
            assert new_command >= 0
            assert new_command < 27
            self._command = new_command
        except AssertionError:
            print("ERROR: Command must be set between 0 and 27")

    command = property(_get_command, _set_command)

    def _get_discomode(self):
        return self._discomode

    def _set_discomode(self, new_discomode):
        try:
            assert new_discomode >= 0xB0
            assert new_discomode <= 0xB8
            self._discomode = new_discomode
        except AssertionError:
            print("ERROR: Discomode must be set between 0xB0 and 0xB8")

    discomode = property(_get_discomode, _set_discomode)

    # Property de conversion RF
    def _get_color_rf(self):
        '''# Retourne la valeur de Color au format RF
        Valeur haute = FF
        Valeur basse = 00
        En RF Va de 00 ==> C8 à F0 ==> D8
        Formule de transformation UDP ==> RF
        (200 - UDP) % 256
        '''
        rf_value =  200 - self.color
        rf_value = rf_value % 256
        return rf_value

    color_rf = property(_get_color_rf)

    def _get_brightness_rf(self):
        '''
        Retourne la valeur de brightness au format RF (string hexa)
        Valeur haute = 1B
        Valeur basse = 02
        '''
        rf_value = (136-(self.brightness + 1) * 8)
        rf_value = rf_value % 256
        rf_value += self.group
        return rf_value

    brightness_rf = property(_get_brightness_rf)

    def _get_color_name(self):
        color_new = self.color - (self.color % 16)
        return MilightRemoteRGBW._COLOR_NAMES[color_new]

    def _set_color_name(self, str_color):
        set_color = False
        for key, value in MilightRemoteRGBW._COLOR_NAMES.items():
            if value == str_color:
                self.command = 15
                self.color = key
                set_color = True
        try:
            assert set_color == True
        except AssertionError:
            print("ERROR: Unknow color name")

    color_name = property(_get_color_name)

    def white_mode(self):
        '''
        Passe les lampes en mode white
        Le code envoyé correspond à la valeur d'un PowerOn augmentée de 16
        '''
        self.command = 17 + (2 * self.group)
        self.mode = 0

    def disco_mode(self):
        '''
        Incrémente la variable discomode à chaque appel, puis set le mode correctement
        '''
        if self.discomode == 0xB8:
            self.discomode = 0xB0
        else:
            self.discomode += 1
        self.command = 13

    def hex_rf_repr(self):
        '''
        Retourne la représentation hexadécimale des commandes radio à envoyer
        '''
        self.increm_sequence()
        StrDiscoMode = '{:02x}'.format(self.discomode)
        StrRemoteID = '{:04x}'.format(self._remoteid)
        StrColor = '{:02x}'.format(self.color_rf)
        StrBrightness = '{:02x}'.format(self.brightness_rf)
        StrCommand = '{:02x}'.format(self.command)
        StrSequence = '{:02x}'.format(self.sequence)
        return "{} {} {} {} {} {} {}".format(StrDiscoMode[0:2].upper(), StrRemoteID[0:2].upper(), StrRemoteID[2:4].upper(), StrColor.upper(), StrBrightness.upper(), StrCommand.upper(), StrSequence.upper())

    #Représentation texte
    def __repr__(self):
        '''
        Renvoie la représentation hexa RF si on appelle l'objet comme si c'était une string
        '''
        return self.hex_rf_repr()

    #Représentation texte
    def __str__(self):
        '''
        Renvoie la représentation hexa RF si on appelle l'objet comme si c'était une string
        '''
        return self.hex_rf_repr()

    #Fonctions d'action
    def power_on(self):
        '''
        Envoie la commande PowerOn pour le groupe en cours
        '''
        self.command = 1 + (2 * self.group)

    def power_off(self):
        '''
        Envoie la commande PowerOff pour le groupe en cours
        '''
        self.command = 2 + (2 * self.group)

    def push_rf(self, delay = 0):
        '''
        Envoie la commande au driver RF
        Chaine Hexa en majuscules de 7 octets séparée par un espace
        XX XX XX XX XX XX XX
        '''
        self.driver.push_command(self.hex_rf_repr(), delay)
